<?php
/**
 * 数据同步服务器设置脚本
 * 
 * 用于初始化服务器数据目录和示例文件
 * 运行此脚本来快速设置服务器环境
 */

// 服务器数据目录配置
define('SERVER_DATA_DIR', __DIR__ . '/server_data');
define('SERVER_CONFIG_PATH', SERVER_DATA_DIR . '/config_data.yaml');
define('SERVER_MINIMAPS_DIR', SERVER_DATA_DIR . '/minimaps');
define('SERVER_MONSTERS_DIR', SERVER_DATA_DIR . '/monsters');

/**
 * 创建目录结构
 */
function createDirectoryStructure() {
    $directories = [
        SERVER_DATA_DIR,
        SERVER_MINIMAPS_DIR,
        SERVER_MONSTERS_DIR,
        SERVER_MINIMAPS_DIR . '/example_map',
        SERVER_MONSTERS_DIR . '/example_monster'
    ];
    
    foreach ($directories as $dir) {
        if (!is_dir($dir)) {
            if (mkdir($dir, 0755, true)) {
                echo "✅ 创建目录: $dir\n";
            } else {
                echo "❌ 创建目录失败: $dir\n";
                return false;
            }
        } else {
            echo "📁 目录已存在: $dir\n";
        }
    }
    
    return true;
}

/**
 * 创建示例配置文件
 */
function createExampleConfig() {
    if (file_exists(SERVER_CONFIG_PATH)) {
        echo "📄 配置文件已存在: " . SERVER_CONFIG_PATH . "\n";
        return true;
    }
    
    $exampleConfig = <<<YAML
# Artale 数据同步配置文件示例
# 请根据实际情况修改此文件

# 英文名到中文名的映射
eng_to_cn:
  example_map: "示例地图"
  example_monster: "示例怪物"
  ant_cave_2: "蚂蚁洞穴2层"
  fire_land_2: "火焰之地2层"
  lost_time_1: "失落的时间1层"
  north_forest_training_ground_2: "北方森林训练场2层"
  north_forest_training_ground_8: "北方森林训练场8层"
  cloud_balcony: "云之阳台"
  green_mushroom: "绿蘑菇"
  mushroom: "蘑菇"
  pig: "猪"
  wild_boar: "野猪"
  fire_boar: "火野猪"
  stone_golem: "石头人"
  skeleton_soldier: "骷髅兵"
  skeleton_officer: "骷髅军官"

# 地图与怪物的绑定关系
map_mobs_mapping:
  example_map:
    - "example_monster"
  ant_cave_2:
    - "green_mushroom"
    - "mushroom"
  fire_land_2:
    - "fire_boar"
    - "stone_golem"
  lost_time_1:
    - "skeleton_soldier"
    - "skeleton_officer"
  north_forest_training_ground_2:
    - "pig"
    - "wild_boar"
  north_forest_training_ground_8:
    - "pig"
    - "wild_boar"
  cloud_balcony:
    - "green_mushroom"
    - "mushroom"
YAML;
    
    if (file_put_contents(SERVER_CONFIG_PATH, $exampleConfig)) {
        echo "✅ 创建示例配置文件: " . SERVER_CONFIG_PATH . "\n";
        return true;
    } else {
        echo "❌ 创建配置文件失败: " . SERVER_CONFIG_PATH . "\n";
        return false;
    }
}

/**
 * 创建示例文件
 */
function createExampleFiles() {
    // 创建示例说明文件
    $readmeContent = <<<README
# 数据同步服务器数据目录

这个目录包含了数据同步服务器的所有数据文件。

## 目录结构

- config_data.yaml: 配置文件，包含地图和怪物的映射关系
- minimaps/: 小地图文件目录
  - 每个子目录代表一个地图
  - 支持 .png, .jpg, .jpeg 格式的图片文件
- monsters/: 怪物文件目录
  - 每个子目录代表一种怪物
  - 支持 .png, .jpg, .jpeg 格式的图片文件

## 使用说明

1. 将地图文件放入 minimaps/ 对应的子目录中
2. 将怪物文件放入 monsters/ 对应的子目录中
3. 更新 config_data.yaml 文件中的映射关系
4. 客户端即可通过API同步这些数据

## 注意事项

- 确保文件名不包含特殊字符
- 建议使用PNG格式的图片文件
- 定期备份重要数据
README;
    
    $readmePath = SERVER_DATA_DIR . '/README.md';
    if (file_put_contents($readmePath, $readmeContent)) {
        echo "✅ 创建说明文件: $readmePath\n";
    }
    
    // 创建示例占位文件
    $placeholderContent = "请将实际的图片文件放置在此目录中";
    
    $exampleMapPlaceholder = SERVER_MINIMAPS_DIR . '/example_map/placeholder.txt';
    if (file_put_contents($exampleMapPlaceholder, $placeholderContent)) {
        echo "✅ 创建示例地图占位文件: $exampleMapPlaceholder\n";
    }
    
    $exampleMonsterPlaceholder = SERVER_MONSTERS_DIR . '/example_monster/placeholder.txt';
    if (file_put_contents($exampleMonsterPlaceholder, $placeholderContent)) {
        echo "✅ 创建示例怪物占位文件: $exampleMonsterPlaceholder\n";
    }
}

/**
 * 检查PHP环境
 */
function checkPhpEnvironment() {
    echo "\n=== PHP环境检查 ===\n";
    
    echo "PHP版本: " . PHP_VERSION . "\n";
    
    $requiredExtensions = ['json', 'hash'];
    $optionalExtensions = ['yaml'];
    
    echo "\n必需扩展检查:\n";
    foreach ($requiredExtensions as $ext) {
        if (extension_loaded($ext)) {
            echo "✅ $ext: 已安装\n";
        } else {
            echo "❌ $ext: 未安装 (必需)\n";
        }
    }
    
    echo "\n可选扩展检查:\n";
    foreach ($optionalExtensions as $ext) {
        if (extension_loaded($ext)) {
            echo "✅ $ext: 已安装\n";
        } else {
            echo "⚠️ $ext: 未安装 (可选，建议安装以获得更好的YAML支持)\n";
        }
    }
    
    echo "\n权限检查:\n";
    if (is_writable(__DIR__)) {
        echo "✅ 当前目录可写\n";
    } else {
        echo "❌ 当前目录不可写\n";
    }
}

/**
 * 显示部署信息
 */
function showDeploymentInfo() {
    echo "\n=== 部署信息 ===\n";
    echo "服务器根目录: " . __DIR__ . "\n";
    echo "数据目录: " . SERVER_DATA_DIR . "\n";
    echo "配置文件: " . SERVER_CONFIG_PATH . "\n";
    echo "小地图目录: " . SERVER_MINIMAPS_DIR . "\n";
    echo "怪物目录: " . SERVER_MONSTERS_DIR . "\n";
    
    echo "\n=== API端点 ===\n";
    $baseUrl = 'http://your-domain.com';
    echo "主页: $baseUrl/index.php\n";
    echo "检查更新: POST $baseUrl/api/sync/check\n";
    echo "服务器状态: GET $baseUrl/api/sync/status\n";
    echo "下载配置: GET $baseUrl/download.php?type=config\n";
    echo "下载地图: GET $baseUrl/download.php?type=minimap&map=<map_name>&file=<file_name>\n";
    echo "下载怪物: GET $baseUrl/download.php?type=monster&monster=<monster_name>&file=<file_name>\n";
    
    echo "\n=== 下一步操作 ===\n";
    echo "1. 将实际的地图文件放入 minimaps/ 目录\n";
    echo "2. 将实际的怪物文件放入 monsters/ 目录\n";
    echo "3. 更新 config_data.yaml 文件\n";
    echo "4. 配置Web服务器指向此目录\n";
    echo "5. 在客户端配置中设置正确的服务器地址\n";
}

// 主程序
echo "Artale 数据同步服务器设置\n";
echo "==============================\n";

// 检查PHP环境
checkPhpEnvironment();

echo "\n=== 开始设置 ===\n";

// 创建目录结构
if (!createDirectoryStructure()) {
    echo "❌ 设置失败：无法创建目录结构\n";
    exit(1);
}

// 创建示例配置文件
if (!createExampleConfig()) {
    echo "❌ 设置失败：无法创建配置文件\n";
    exit(1);
}

// 创建示例文件
createExampleFiles();

echo "\n✅ 设置完成！\n";

// 显示部署信息
showDeploymentInfo();

echo "\n🎉 数据同步服务器已准备就绪！\n";
?>