<?php
/**
 * Artale 数据同步服务器 - PHP版本
 * 
 * 这是一个PHP实现的数据同步API服务器
 * 支持地图、怪物数据和配置文件的同步功能
 * 
 * 部署要求:
 * - PHP 7.4+
 * - 支持JSON扩展
 * - 文件读写权限
 * 
 * 使用方法:
 * 1. 将此文件放置在Web服务器目录中
 * 2. 创建server_data目录结构
 * 3. 配置Web服务器指向此目录
 * 4. 访问 http://your-domain.com/index.php
 */

header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

// 处理OPTIONS预检请求
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

// 服务器数据目录配置
define('SERVER_DATA_DIR', __DIR__ . '/server_data');
define('SERVER_CONFIG_PATH', SERVER_DATA_DIR . '/config_data.yaml');
define('SERVER_MINIMAPS_DIR', SERVER_DATA_DIR . '/minimaps');
define('SERVER_MONSTERS_DIR', SERVER_DATA_DIR . '/monsters');

// 创建必要的目录
if (!is_dir(SERVER_DATA_DIR)) {
    mkdir(SERVER_DATA_DIR, 0755, true);
}
if (!is_dir(SERVER_MINIMAPS_DIR)) {
    mkdir(SERVER_MINIMAPS_DIR, 0755, true);
}
if (!is_dir(SERVER_MONSTERS_DIR)) {
    mkdir(SERVER_MONSTERS_DIR, 0755, true);
}

/**
 * 计算文件的MD5哈希值
 */
function calculateFileHash($filePath) {
    if (!file_exists($filePath)) {
        return '';
    }
    return md5_file($filePath);
}

/**
 * 解析YAML文件（简单实现）
 */
function parseYaml($filePath) {
    if (!file_exists($filePath)) {
        return [];
    }
    
    // 如果有yaml扩展，使用它
    if (function_exists('yaml_parse_file')) {
        return yaml_parse_file($filePath);
    }
    
    // 简单的YAML解析（仅支持基本格式）
    $content = file_get_contents($filePath);
    $lines = explode("\n", $content);
    $result = [];
    $currentSection = null;
    
    foreach ($lines as $line) {
        $line = trim($line);
        if (empty($line) || strpos($line, '#') === 0) {
            continue;
        }
        
        if (preg_match('/^([a-zA-Z_][a-zA-Z0-9_]*):$/', $line, $matches)) {
            $currentSection = $matches[1];
            $result[$currentSection] = [];
        } elseif (preg_match('/^  ([a-zA-Z_][a-zA-Z0-9_]*): (.+)$/', $line, $matches)) {
            if ($currentSection) {
                $value = trim($matches[2], '"\' ');
                if ($value === 'true') $value = true;
                elseif ($value === 'false') $value = false;
                elseif (is_numeric($value)) $value = (float)$value;
                $result[$currentSection][$matches[1]] = $value;
            }
        }
    }
    
    return $result;
}

/**
 * 获取服务器端数据信息
 */
function getServerDataInfo() {
    $serverInfo = [
        'config_data' => [],
        'minimaps' => [],
        'monsters' => [],
        'last_update_time' => date('c')
    ];
    
    // 读取配置数据
    if (file_exists(SERVER_CONFIG_PATH)) {
        $configData = parseYaml(SERVER_CONFIG_PATH);
        $serverInfo['config_data'] = [
            'hash' => calculateFileHash(SERVER_CONFIG_PATH),
            'download_url' => getBaseUrl() . '/download.php?type=config',
            'maps' => isset($configData['map_mobs_mapping']) ? array_keys($configData['map_mobs_mapping']) : [],
            'monsters' => isset($configData['eng_to_cn']) ? array_keys($configData['eng_to_cn']) : []
        ];
    }
    
    // 扫描小地图文件
    if (is_dir(SERVER_MINIMAPS_DIR)) {
        $mapDirs = scandir(SERVER_MINIMAPS_DIR);
        foreach ($mapDirs as $mapDir) {
            if ($mapDir === '.' || $mapDir === '..') continue;
            
            $mapPath = SERVER_MINIMAPS_DIR . '/' . $mapDir;
            if (is_dir($mapPath)) {
                $mapFiles = [];
                $files = scandir($mapPath);
                foreach ($files as $file) {
                    if ($file === '.' || $file === '..') continue;
                    
                    $filePath = $mapPath . '/' . $file;
                    if (is_file($filePath) && preg_match('/\.(png|jpg|jpeg)$/i', $file)) {
                        $mapFiles[$file] = [
                            'hash' => calculateFileHash($filePath),
                            'download_url' => getBaseUrl() . '/download.php?type=minimap&map=' . urlencode($mapDir) . '&file=' . urlencode($file)
                        ];
                    }
                }
                if (!empty($mapFiles)) {
                    $serverInfo['minimaps'][$mapDir] = ['files' => $mapFiles];
                }
            }
        }
    }
    
    // 扫描怪物文件
    if (is_dir(SERVER_MONSTERS_DIR)) {
        $monsterDirs = scandir(SERVER_MONSTERS_DIR);
        foreach ($monsterDirs as $monsterDir) {
            if ($monsterDir === '.' || $monsterDir === '..') continue;
            
            $monsterPath = SERVER_MONSTERS_DIR . '/' . $monsterDir;
            if (is_dir($monsterPath)) {
                $monsterFiles = [];
                $files = scandir($monsterPath);
                foreach ($files as $file) {
                    if ($file === '.' || $file === '..') continue;
                    
                    $filePath = $monsterPath . '/' . $file;
                    if (is_file($filePath) && preg_match('/\.(png|jpg|jpeg)$/i', $file)) {
                        $monsterFiles[$file] = [
                            'hash' => calculateFileHash($filePath),
                            'download_url' => getBaseUrl() . '/download.php?type=monster&monster=' . urlencode($monsterDir) . '&file=' . urlencode($file)
                        ];
                    }
                }
                if (!empty($monsterFiles)) {
                    $serverInfo['monsters'][$monsterDir] = ['files' => $monsterFiles];
                }
            }
        }
    }
    
    return $serverInfo;
}

/**
 * 比较客户端和服务器端数据
 */
function compareDataInfo($clientInfo, $serverInfo) {
    $updates = [
        'has_updates' => false,
        'config_data' => null,
        'minimaps' => [],
        'monsters' => []
    ];
    
    // 检查配置数据
    $clientConfig = isset($clientInfo['config_data']) ? $clientInfo['config_data'] : [];
    $serverConfig = isset($serverInfo['config_data']) ? $serverInfo['config_data'] : [];
    
    if (!empty($serverConfig['hash']) && 
        (empty($clientConfig['hash']) || $clientConfig['hash'] !== $serverConfig['hash'])) {
        $updates['config_data'] = $serverConfig;
        $updates['has_updates'] = true;
    }
    
    // 检查小地图文件
    $clientMinimaps = isset($clientInfo['minimaps']) ? $clientInfo['minimaps'] : [];
    $serverMinimaps = isset($serverInfo['minimaps']) ? $serverInfo['minimaps'] : [];
    
    foreach ($serverMinimaps as $mapName => $serverMapInfo) {
        $clientMapInfo = isset($clientMinimaps[$mapName]) ? $clientMinimaps[$mapName] : [];
        $serverFiles = isset($serverMapInfo['files']) ? $serverMapInfo['files'] : [];
        
        $filesToUpdate = [];
        foreach ($serverFiles as $fileName => $serverFileInfo) {
            $clientFileHash = isset($clientMapInfo[$fileName]) ? $clientMapInfo[$fileName] : '';
            $serverFileHash = isset($serverFileInfo['hash']) ? $serverFileInfo['hash'] : '';
            
            if ($clientFileHash !== $serverFileHash) {
                $filesToUpdate[$fileName] = $serverFileInfo;
            }
        }
        
        if (!empty($filesToUpdate)) {
            $updates['minimaps'][$mapName] = ['files' => $filesToUpdate];
            $updates['has_updates'] = true;
        }
    }
    
    // 检查怪物文件
    $clientMonsters = isset($clientInfo['monsters']) ? $clientInfo['monsters'] : [];
    $serverMonsters = isset($serverInfo['monsters']) ? $serverInfo['monsters'] : [];
    
    foreach ($serverMonsters as $monsterName => $serverMonsterInfo) {
        $clientMonsterInfo = isset($clientMonsters[$monsterName]) ? $clientMonsters[$monsterName] : [];
        $serverFiles = isset($serverMonsterInfo['files']) ? $serverMonsterInfo['files'] : [];
        
        $filesToUpdate = [];
        foreach ($serverFiles as $fileName => $serverFileInfo) {
            $clientFileHash = isset($clientMonsterInfo[$fileName]) ? $clientMonsterInfo[$fileName] : '';
            $serverFileHash = isset($serverFileInfo['hash']) ? $serverFileInfo['hash'] : '';
            
            if ($clientFileHash !== $serverFileHash) {
                $filesToUpdate[$fileName] = $serverFileInfo;
            }
        }
        
        if (!empty($filesToUpdate)) {
            $updates['monsters'][$monsterName] = ['files' => $filesToUpdate];
            $updates['has_updates'] = true;
        }
    }
    
    return $updates;
}

/**
 * 获取基础URL
 */
function getBaseUrl() {
    $protocol = isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? 'https' : 'http';
    $host = $_SERVER['HTTP_HOST'];
    $path = dirname($_SERVER['SCRIPT_NAME']);
    return $protocol . '://' . $host . $path;
}

/**
 * 发送JSON响应
 */
function sendJsonResponse($data, $statusCode = 200) {
    http_response_code($statusCode);
    echo json_encode($data, JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT);
    exit();
}

/**
 * 发送错误响应
 */
function sendErrorResponse($message, $statusCode = 500) {
    sendJsonResponse(['error' => $message], $statusCode);
}

// 路由处理
$requestUri = $_SERVER['REQUEST_URI'];
$requestMethod = $_SERVER['REQUEST_METHOD'];
$parsedUrl = parse_url($requestUri);
$path = $parsedUrl['path'];
$query = isset($parsedUrl['query']) ? $parsedUrl['query'] : '';

// 解析查询参数
parse_str($query, $queryParams);
$action = isset($queryParams['action']) ? $queryParams['action'] : null;

// 移除脚本名称部分
$scriptName = $_SERVER['SCRIPT_NAME'];
$basePath = dirname($scriptName);
if ($basePath !== '/') {
    $path = str_replace($basePath, '', $path);
}

try {
    // API路由
    if (($path === '/api/sync/check' || $action === 'check') && $requestMethod === 'POST') {
        // 检查更新API
        $input = file_get_contents('php://input');
        $clientInfo = json_decode($input, true);
        
        if (!$clientInfo) {
            sendErrorResponse('无效的请求数据', 400);
        }
        
        $serverInfo = getServerDataInfo();
        $updates = compareDataInfo($clientInfo, $serverInfo);
        
        sendJsonResponse($updates);
        
    } elseif (($path === '/api/sync/status' || $action === 'status') && $requestMethod === 'GET') {
        // 获取同步状态
        $serverInfo = getServerDataInfo();
        $status = [
            'server_time' => date('c'),
            'maps_count' => count($serverInfo['minimaps']),
            'monsters_count' => count($serverInfo['monsters']),
            'config_available' => !empty($serverInfo['config_data'])
        ];
        
        sendJsonResponse($status);
        
    } elseif (($path === '/' || $path === '/index.php') && !$action) {
        // 主页
        $serverInfo = getServerDataInfo();
        $status = [
            'message' => 'Artale 数据同步服务器 - PHP版本',
            'version' => '1.0.0',
            'server_time' => date('c'),
            'endpoints' => [
                'POST /api/sync/check' => '检查更新',
                'GET /api/sync/status' => '获取服务器状态',
                'GET /download.php?type=config' => '下载配置文件',
                'GET /download.php?type=minimap&map=<map_name>&file=<file_name>' => '下载小地图文件',
                'GET /download.php?type=monster&monster=<monster_name>&file=<file_name>' => '下载怪物文件'
            ],
            'data_summary' => [
                'maps_count' => count($serverInfo['minimaps']),
                'monsters_count' => count($serverInfo['monsters']),
                'config_available' => !empty($serverInfo['config_data'])
            ]
        ];
        
        sendJsonResponse($status);
        
    } else {
        sendErrorResponse('API端点未找到', 404);
    }
    
} catch (Exception $e) {
    sendErrorResponse('服务器内部错误: ' . $e->getMessage(), 500);
}
?>