<?php
/**
 * 文件下载处理脚本
 * 
 * 处理配置文件、小地图文件和怪物文件的下载请求
 * 
 * 支持的下载类型:
 * - config: 配置文件
 * - minimap: 小地图文件
 * - monster: 怪物文件
 */

// 服务器数据目录配置
define('SERVER_DATA_DIR', __DIR__ . '/server_data');
define('SERVER_CONFIG_PATH', SERVER_DATA_DIR . '/config_data.yaml');
define('SERVER_MINIMAPS_DIR', SERVER_DATA_DIR . '/minimaps');
define('SERVER_MONSTERS_DIR', SERVER_DATA_DIR . '/monsters');

/**
 * 发送错误响应
 */
function sendError($message, $code = 404) {
    http_response_code($code);
    header('Content-Type: application/json; charset=utf-8');
    echo json_encode(['error' => $message], JSON_UNESCAPED_UNICODE);
    exit();
}

/**
 * 安全地获取文件路径
 */
function getSafeFilePath($basePath, $relativePath) {
    // 防止目录遍历攻击
    $relativePath = str_replace(['../', '..\\', '..'], '', $relativePath);
    $fullPath = realpath($basePath . '/' . $relativePath);
    
    // 确保文件在允许的目录内
    if ($fullPath === false || strpos($fullPath, realpath($basePath)) !== 0) {
        return false;
    }
    
    return $fullPath;
}

/**
 * 发送文件下载
 */
function sendFileDownload($filePath, $fileName = null) {
    if (!file_exists($filePath) || !is_file($filePath)) {
        sendError('文件不存在');
    }
    
    if ($fileName === null) {
        $fileName = basename($filePath);
    }
    
    // 获取文件信息
    $fileSize = filesize($filePath);
    // $mimeType = mime_content_type($filePath);
    // 使用 finfo_file 获取 MIME 类型，如果不可用则写死为 application/octet-stream
    if (function_exists('finfo_open')) {
        $finfo = finfo_open(FILEINFO_MIME_TYPE);
        $mimeType = finfo_file($finfo, $filePath);
        finfo_close($finfo);
    } else {
        $mimeType = 'application/octet-stream';
    }
    
    // 设置下载头
    header('Content-Type: ' . $mimeType);
    header('Content-Disposition: attachment; filename="' . $fileName . '"');
    header('Content-Length: ' . $fileSize);
    header('Cache-Control: no-cache, must-revalidate');
    header('Expires: 0');
    
    // 输出文件内容
    if ($fileSize > 0) {
        $handle = fopen($filePath, 'rb');
        if ($handle) {
            while (!feof($handle)) {
                echo fread($handle, 8192);
                flush();
            }
            fclose($handle);
        }
    }
    
    exit();
}

// 获取请求参数
$type = isset($_GET['type']) ? $_GET['type'] : '';
$map = isset($_GET['map']) ? $_GET['map'] : '';
$monster = isset($_GET['monster']) ? $_GET['monster'] : '';
$file = isset($_GET['file']) ? $_GET['file'] : '';

try {
    switch ($type) {
        case 'config':
            // 下载配置文件
            if (!file_exists(SERVER_CONFIG_PATH)) {
                sendError('配置文件不存在');
            }
            sendFileDownload(SERVER_CONFIG_PATH, 'config_data.yaml');
            break;
            
        case 'minimap':
            // 下载小地图文件
            if (empty($map) || empty($file)) {
                sendError('缺少必要参数: map 和 file', 400);
            }
            
            $mapPath = getSafeFilePath(SERVER_MINIMAPS_DIR, $map);
            if (!$mapPath || !is_dir($mapPath)) {
                sendError('地图目录不存在');
            }
            
            $filePath = getSafeFilePath($mapPath, $file);
            if (!$filePath) {
                sendError('文件不存在或路径不安全');
            }
            
            // 验证文件扩展名
            if (!preg_match('/\.(png|jpg|jpeg)$/i', $file)) {
                sendError('不支持的文件类型');
            }
            
            sendFileDownload($filePath, $file);
            break;
            
        case 'monster':
            // 下载怪物文件
            if (empty($monster) || empty($file)) {
                sendError('缺少必要参数: monster 和 file', 400);
            }
            
            $monsterPath = getSafeFilePath(SERVER_MONSTERS_DIR, $monster);
            if (!$monsterPath || !is_dir($monsterPath)) {
                sendError('怪物目录不存在');
            }
            
            $filePath = getSafeFilePath($monsterPath, $file);
            if (!$filePath) {
                sendError('文件不存在或路径不安全');
            }
            
            // 验证文件扩展名
            if (!preg_match('/\.(png|jpg|jpeg)$/i', $file)) {
                sendError('不支持的文件类型');
            }
            
            sendFileDownload($filePath, $file);
            break;
            
        default:
            sendError('无效的下载类型', 400);
    }
    
} catch (Exception $e) {
    sendError('下载失败: ' . $e->getMessage(), 500);
}
?>