#!/bin/bash

# Artale 数据同步服务器 - 快速部署脚本
# 适用于 Ubuntu/Debian 系统

set -e

echo "🚀 Artale 数据同步服务器部署脚本"
echo "======================================"

# 颜色定义
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
NC='\033[0m' # No Color

# 函数定义
print_status() {
    echo -e "${BLUE}[INFO]${NC} $1"
}

print_success() {
    echo -e "${GREEN}[SUCCESS]${NC} $1"
}

print_warning() {
    echo -e "${YELLOW}[WARNING]${NC} $1"
}

print_error() {
    echo -e "${RED}[ERROR]${NC} $1"
}

# 检查是否为root用户
check_root() {
    if [[ $EUID -eq 0 ]]; then
        print_warning "检测到root用户，建议使用sudo运行此脚本"
    fi
}

# 检测操作系统
detect_os() {
    if [[ -f /etc/os-release ]]; then
        . /etc/os-release
        OS=$NAME
        VER=$VERSION_ID
    else
        print_error "无法检测操作系统"
        exit 1
    fi
    print_status "检测到操作系统: $OS $VER"
}

# 更新系统包
update_system() {
    print_status "更新系统包..."
    sudo apt update
    sudo apt upgrade -y
    print_success "系统包更新完成"
}

# 安装PHP和必要扩展
install_php() {
    print_status "安装PHP和必要扩展..."
    
    # 安装PHP和常用扩展
    sudo apt install -y php php-fpm php-cli php-json php-mbstring php-curl php-yaml
    
    # 检查PHP版本
    PHP_VERSION=$(php -v | head -n1 | cut -d' ' -f2 | cut -d'.' -f1,2)
    print_success "PHP $PHP_VERSION 安装完成"
    
    # 启用PHP-FPM
    sudo systemctl enable php*-fpm
    sudo systemctl start php*-fpm
    print_success "PHP-FPM 服务已启动"
}

# 安装Web服务器
install_webserver() {
    echo
    echo "请选择Web服务器:"
    echo "1) Apache"
    echo "2) Nginx"
    read -p "请输入选择 (1-2): " choice
    
    case $choice in
        1)
            install_apache
            ;;
        2)
            install_nginx
            ;;
        *)
            print_error "无效选择"
            exit 1
            ;;
    esac
}

# 安装Apache
install_apache() {
    print_status "安装Apache..."
    sudo apt install -y apache2 libapache2-mod-php
    
    # 启用必要模块
    sudo a2enmod rewrite
    sudo a2enmod headers
    sudo a2enmod php*
    
    # 启动Apache
    sudo systemctl enable apache2
    sudo systemctl start apache2
    
    print_success "Apache 安装完成"
    WEB_SERVER="apache"
}

# 安装Nginx
install_nginx() {
    print_status "安装Nginx..."
    sudo apt install -y nginx
    
    # 启动Nginx
    sudo systemctl enable nginx
    sudo systemctl start nginx
    
    print_success "Nginx 安装完成"
    WEB_SERVER="nginx"
}

# 配置防火墙
setup_firewall() {
    print_status "配置防火墙..."
    
    if command -v ufw &> /dev/null; then
        sudo ufw allow 22/tcp
        sudo ufw allow 80/tcp
        sudo ufw allow 443/tcp
        sudo ufw --force enable
        print_success "UFW防火墙配置完成"
    else
        print_warning "未检测到UFW，请手动配置防火墙"
    fi
}

# 部署应用文件
deploy_application() {
    print_status "部署应用文件..."
    
    # 确定Web根目录
    if [[ $WEB_SERVER == "apache" ]]; then
        WEB_ROOT="/var/www/html"
    else
        WEB_ROOT="/var/www/html"
    fi
    
    # 创建应用目录
    APP_DIR="$WEB_ROOT/artale-sync"
    sudo mkdir -p $APP_DIR
    
    # 复制文件
    sudo cp *.php $APP_DIR/
    sudo cp .htaccess $APP_DIR/ 2>/dev/null || true
    sudo cp README.md $APP_DIR/
    
    # 设置权限
    sudo chown -R www-data:www-data $APP_DIR
    sudo chmod -R 755 $APP_DIR
    
    print_success "应用文件部署完成: $APP_DIR"
    
    # 运行设置脚本
    print_status "运行应用设置..."
    cd $APP_DIR
    sudo -u www-data php setup.php
    
    print_success "应用设置完成"
}

# 配置Web服务器
configure_webserver() {
    if [[ $WEB_SERVER == "nginx" ]]; then
        configure_nginx
    fi
}

# 配置Nginx
configure_nginx() {
    print_status "配置Nginx..."
    
    # 创建站点配置
    sudo tee /etc/nginx/sites-available/artale-sync > /dev/null <<EOF
server {
    listen 80;
    server_name _;
    root /var/www/html/artale-sync;
    index index.php;
    
    access_log /var/log/nginx/artale_sync_access.log;
    error_log /var/log/nginx/artale_sync_error.log;
    
    add_header X-Content-Type-Options nosniff;
    add_header X-Frame-Options DENY;
    add_header X-XSS-Protection "1; mode=block";
    add_header Access-Control-Allow-Origin *;
    add_header Access-Control-Allow-Methods "GET, POST, OPTIONS";
    add_header Access-Control-Allow-Headers "Content-Type, Authorization";
    
    if (\$request_method = 'OPTIONS') {
        return 200;
    }
    
    location ~ ^/api/sync/(check|status)\$ {
        try_files \$uri /index.php\$is_args\$args;
    }
    
    location = / {
        try_files \$uri /index.php\$is_args\$args;
    }
    
    location ~ \.php\$ {
        try_files \$uri =404;
        fastcgi_pass unix:/var/run/php/php*-fpm.sock;
        fastcgi_index index.php;
        fastcgi_param SCRIPT_FILENAME \$document_root\$fastcgi_script_name;
        include fastcgi_params;
    }
    
    location ~* \.(png|jpg|jpeg|gif|ico|svg)\$ {
        expires 1M;
        add_header Cache-Control "public, immutable";
        try_files \$uri =404;
    }
    
    location ~ /\. {
        deny all;
    }
    
    location ~ \.(yaml|yml)\$ {
        deny all;
    }
    
    location ~ ^/server_data/ {
        internal;
    }
}
EOF
    
    # 启用站点
    sudo ln -sf /etc/nginx/sites-available/artale-sync /etc/nginx/sites-enabled/
    sudo rm -f /etc/nginx/sites-enabled/default
    
    # 测试配置
    sudo nginx -t
    sudo systemctl reload nginx
    
    print_success "Nginx配置完成"
}

# 显示部署信息
show_deployment_info() {
    echo
    echo "🎉 部署完成！"
    echo "======================================"
    echo
    echo "服务器信息:"
    echo "- Web服务器: $WEB_SERVER"
    echo "- PHP版本: $PHP_VERSION"
    echo "- 应用目录: $APP_DIR"
    echo
    echo "访问地址:"
    echo "- 主页: http://$(hostname -I | awk '{print $1}')/artale-sync/"
    echo "- API状态: http://$(hostname -I | awk '{print $1}')/artale-sync/api/sync/status"
    echo
    echo "下一步操作:"
    echo "1. 将地图文件上传到: $APP_DIR/server_data/minimaps/"
    echo "2. 将怪物文件上传到: $APP_DIR/server_data/monsters/"
    echo "3. 编辑配置文件: $APP_DIR/server_data/config_data.yaml"
    echo "4. 在客户端配置中设置服务器地址"
    echo
    echo "管理命令:"
    echo "- 重启Web服务器: sudo systemctl restart $WEB_SERVER"
    echo "- 查看错误日志: sudo tail -f /var/log/$WEB_SERVER/error.log"
    echo "- 查看访问日志: sudo tail -f /var/log/$WEB_SERVER/access.log"
    echo
}

# 主函数
main() {
    check_root
    detect_os
    
    echo
    read -p "是否继续安装? (y/N): " confirm
    if [[ ! $confirm =~ ^[Yy]$ ]]; then
        echo "安装已取消"
        exit 0
    fi
    
    update_system
    install_php
    install_webserver
    setup_firewall
    deploy_application
    configure_webserver
    show_deployment_info
}

# 运行主函数
main "$@"